define([
    'App',
    'underscore',
    'models/abstract-collection',
    'modules/new-appointment-request/resources/common/city-state/model',
], function(app, _, Collection, Model) {
    'use strict';


    /**
     * @class CityStateCollection
     * @typedef {Backbone.Collection}
     */
    return Collection.extend({
        resourceName: 'city-state',
        model: Model,

        url: function() {
            var resource = this.getResource();
            return resource.formatHref({});
        },

        fetch: function() {
            var options = {data: this._getFetchData()};

            return Collection.prototype.fetch.call(this, options);
        },

        /**
         * Sorts the collection alphabetically with primary group being state
         * and the secondary group being city
         * @param {Backbone.Model} lhs
         * @param {Backbone.Model} rhs
         * @return {Number}
         */
        comparator: function(lhs, rhs) {
            return lhs.compare(rhs);
        },

        /**
         * Converts the collection into an object.
         * key: The full state name
         * value: An array of cities in that state
         * @return {Object}
         */
        group: function() {
            var obj = {};
            this.each(function(model) {
                var stateName = model.getStateName();
                var city = model.get('city');
                var state = model.get('stateAbbrev');
                var title = _.compact([city, state]).join(', ');

                if (!obj.hasOwnProperty(stateName)) {
                    obj[stateName] = [];
                }
                obj[stateName].push({text: title, value: model.id});
            });
            return obj;
        },

        /**
         * Setup the attributes for the collection fetch
         * Creates the data to fetch patient facilities
         * @return {*} The get request attributes
         * @private
         */
        _getFetchData: function() {
            var siteCode = app.facilities.map(this._mapCodes);

            return {
                'facility-code': siteCode,
            };
        },

        /**
         * A shortcut function to avoid nesting in _getFetchData
         * To be used a part of _.map
         * @param {Backbone.Model} model
         * @return {string}
         * @private
         */
        _mapCodes: function(model) {
            return model.get('facilityCode');
        },
    });
});
